/*
 * RotateSegmentMovementInfo.h
 *
 * Created 9/22/2009 By Johnny Huynh
 *
 * Version 00.00.01 9/22/2009
 *
 * Copyright Information:
 * All content copyright  2009 Johnny Huynh. All rights reserved.
 */
 
 /**
  * Contains information for rotating a segment about its origin.
  */
 
 #ifndef ROTATE_SEGMENT_MOVEMENT_INFO_H
 #define ROTATE_SEGMENT_MOVEMENT_INFO_H
 
 template <typename T> class RotateSegmentMovementInfo;
 
 #include "global.h"
 
 #include "CurveExactMovementInfo.h"
 #include "collisionSegment.h"
 #include "pointerTo.h"
 
 /**
  * Class specification for RotateSegmentMovementInfo
  */
 template <typename T>
 class RotateSegmentMovementInfo : public CurveExactMovementInfo<T>
 {
 // Data Members
 private:
    PT(CollisionSegment) _collision_segment_Ptr;
    
 // Local Functions
 public:
    RotateSegmentMovementInfo( CollisionSegment* collision_segment_Ptr,
                            const VECTOR3_TYPE& acceleration = VECTOR3_TYPE( ZERO, ZERO, ZERO ), 
                            const VECTOR3_TYPE& velocity = VECTOR3_TYPE( ZERO, ZERO, ZERO ),
                            const VECTOR3_TYPE& begin_point = VECTOR3_TYPE( ZERO, ZERO, ZERO ),
                            const T distance = ONE, const double duration = 0.0 );
    RotateSegmentMovementInfo( const RotateSegmentMovementInfo<T>& rotate_segment_movement_info );
    virtual ~RotateSegmentMovementInfo();
    inline RotateSegmentMovementInfo<T>& operator=( const RotateSegmentMovementInfo<T>& rotate_segment_movement_info );
    virtual inline double process_movement( Object<T>* obj_Ptr, MoveInfo<T>* move_info_Ptr, double current_time );
 
 // Private Functions
 private:
    
 // Public Static Functions
 public:
    
 };
 
 /** LOCAL FUNCTIONS **/
 
 /**
  * Constructor
  * The direction of the specified segment will trace the arc specified for CurveExactMovementInfo:
  * P(t) = (distance*(acceleration*t^2 + velocity*t)) + begin_point
  */
 template <typename T>
 RotateSegmentMovementInfo<T>::RotateSegmentMovementInfo( CollisionSegment* collision_segment_Ptr,
                                                    const VECTOR3_TYPE& acceleration, 
                                                    const VECTOR3_TYPE& velocity,
                                                    const VECTOR3_TYPE& begin_point,
                                                    const T distance, const double duration )
                           : CurveExactMovementInfo<T>( acceleration, velocity, begin_point, distance, duration ),
                             _collision_segment_Ptr( collision_segment_Ptr )
 {
    nassertv( _collision_segment_Ptr != NULL );
 }
 
 /**
  * Copy Constructor
  */
 template <typename T>
 RotateSegmentMovementInfo<T>::RotateSegmentMovementInfo( const RotateSegmentMovementInfo<T>& rotate_segment_movement_info )
                      : CurveExactMovementInfo<T>( rotate_segment_movement_info ),
                        _collision_segment_Ptr( rotate_segment_movement_info._collision_segment_Ptr )
 {
    
 }
 
 /**
  * Destructor
  */
 template <typename T>
 RotateSegmentMovementInfo<T>::~RotateSegmentMovementInfo()
 {
    
 }
 
 /**
  * operator=() copies the content of the specified RotateSegmentMovementInfo to this RotateSegmentMovementInfo.
  *
  * @param (const RotateSegmentMovementInfo<T>&) rotate_segment_movement_info
  * @return RotateSegmentMovementInfo<T>&
  */
 template <typename T>
 inline RotateSegmentMovementInfo<T>& RotateSegmentMovementInfo<T>::operator=( const RotateSegmentMovementInfo<T>& rotate_segment_movement_info )
 {
    CurveExactMovementInfo<T>::operator=( rotate_segment_movement_info );
    _collision_segment_Ptr = rotate_segment_movement_info._collision_segment_Ptr;
    
    return *this;
 }
 
 /**
  * process_movement() processes the movement specified by this MovementInfo, given the Object to process on,
  * the MoveInfo, and current time. The duration unprocessed returned is greater than zero if this MovementInfo
  * has finished processing, and there is time leftover.
  *
  * @param (Object<T>*) obj_Ptr
  * @param (MoveInfo<T>*) move_info_Ptr
  * @param (double) current_time
  * @return double - the time left unprocessed (i.e. a double greater than zero if this MovementInfo is done processing)
  */
 template <typename T>
 inline double RotateSegmentMovementInfo<T>::process_movement( Object<T>* obj_Ptr, MoveInfo<T>* move_info_Ptr, double current_time )
 {  
    nassertr( obj_Ptr != NULL, MovementInfo<T>::get_duration() );
    
    double move_action_invoke_time( move_info_Ptr->get_time_move_action_was_invoked() );
    double last_process_time( move_info_Ptr->get_time_last_processed_move_task() );
    nassertr( last_process_time >= move_action_invoke_time, MovementInfo<T>::get_duration() );
    
    double duration( MovementInfo<T>::get_duration() );
    double elapse_time_since_invocation( current_time - move_action_invoke_time );
    
    if ( elapse_time_since_invocation > 0.0 && (CurveMovementInfo<T>::get_acceleration() != VECTOR3_TYPE(ZERO,ZERO,ZERO) || CurveMovementInfo<T>::get_velocity() != VECTOR3_TYPE(ZERO,ZERO,ZERO)) )
    {
        // gradually move for only the movement duration (measured in seconds)
        //if ( elapse_time_since_invocation > duration ) // if true, this will be the last time we process this movement_info
        
        double invocation_ratio( (elapse_time_since_invocation >= duration ? 1.0 : elapse_time_since_invocation / duration) );
        double invocation_ratio_squared( invocation_ratio * invocation_ratio );
        
        // move the Object to a specific location
        // (a*invocation_ratio^2 + v*invocation_ratio) + p
        //obj_Ptr->set_pos( (CurveMovementInfo<T>::get_acceleration()*invocation_ratio_squared) + (CurveMovementInfo<T>::get_velocity()*invocation_ratio) + _begin_point );
        _collision_segment_Ptr->set_point_b( (CurveMovementInfo<T>::get_acceleration()*invocation_ratio_squared) + (CurveMovementInfo<T>::get_velocity()*invocation_ratio) + CurveExactMovementInfo<T>::get_begin_point() );
        //printf( "acceleration: (%1f, %2f, %3f)\n", get_acceleration().get_x(), get_acceleration().get_y(), get_acceleration().get_z() );
        //printf( "velocity: (%1f, %2f, %3f)\n", get_velocity().get_x(), get_velocity().get_y(), get_velocity().get_z() );
    }
    
    if ( elapse_time_since_invocation > duration )
        return elapse_time_since_invocation - duration;
    else // elapse_time_since_invocation <= duration
        return 0.0;
 }
 
 /** PUBLIC STATIC FUNCTIONS **/
 
 #endif // ROTATE_SEGMENT_MOVEMENT_INFO_H